<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Monitoring;

final class HealthCheckRepository
{
    /**
     * @param array<string, mixed> $result
     */
    public function add(int $targetId, string $checkType, array $result, int $score): void
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_checks';

        $inserted = $wpdb->insert(
            $table,
            [
                'target_id' => $targetId,
                'check_type' => $checkType,
                'result_json' => wp_json_encode($result),
                'score' => $score,
                'created_at' => current_time('mysql'),
            ],
            ['%d', '%s', '%s', '%d', '%s']
        );

        if ($inserted !== 1) {
            throw new \RuntimeException('Unable to persist health check result.');
        }
    }

    /**
     * @param array<int, int> $targetIds
     * @return array<int, int>
     */
    public function latestScoresByTargetIds(array $targetIds): array
    {
        return $this->latestScoresByTargetIdsByType($targetIds, 'core');
    }

    /**
     * @param array<int, int> $targetIds
     * @return array<int, int>
     */
    public function latestScoresByTargetIdsByType(array $targetIds, string $checkType): array
    {
        if ($targetIds === []) {
            return [];
        }

        global $wpdb;

        $checksTable = $wpdb->prefix . 'wpo_checks';
        $ids = array_values(array_unique(array_map('intval', $targetIds)));
        $idsPlaceholders = implode(',', array_fill(0, count($ids), '%d'));

        $sql = "
            SELECT c.target_id, c.score
            FROM {$checksTable} c
            INNER JOIN (
                SELECT target_id, MAX(id) AS latest_id
                FROM {$checksTable}
                WHERE target_id IN ({$idsPlaceholders}) AND check_type = %s
                GROUP BY target_id
            ) latest ON latest.latest_id = c.id
        ";

        $params = array_merge($ids, [$checkType]);
        $prepared = $wpdb->prepare($sql, ...$params);

        if ($prepared === null) {
            return [];
        }

        $rows = $wpdb->get_results($prepared, ARRAY_A);
        $scores = [];

        if (!is_array($rows)) {
            return $scores;
        }

        foreach ($rows as $row) {
            $scores[(int) $row['target_id']] = (int) $row['score'];
        }

        return $scores;
    }

    /**
     * @return array<string, mixed>|null
     */
    public function latestResultByTargetAndType(int $targetId, string $checkType): ?array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_checks';
        $sql = $wpdb->prepare(
            "SELECT id, target_id, check_type, result_json, score, created_at
             FROM {$table}
             WHERE target_id = %d AND check_type = %s
             ORDER BY id DESC
             LIMIT 1",
            $targetId,
            $checkType
        );

        if ($sql === null) {
            return null;
        }

        $row = $wpdb->get_row($sql, ARRAY_A);

        if (!is_array($row)) {
            return null;
        }

        $decoded = json_decode((string) ($row['result_json'] ?? ''), true);
        $row['result_json'] = is_array($decoded) ? $decoded : [];

        return $row;
    }
}
