<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Alerting;

use WpOrchestrator\Support\Logger;

final class AlertDispatcher
{
    private const DEDUP_TTL = 10 * MINUTE_IN_SECONDS;

    public function __construct(private readonly Logger $logger)
    {
    }

    /**
     * @param array<string, mixed> $target
     * @param array<string, mixed> $result
     */
    public function dispatchIfNeeded(array $target, array $result, int $score): void
    {
        $targetId = (int) ($target['id'] ?? 0);
        $targetName = (string) ($target['name'] ?? 'unknown');
        $status = (string) ($result['status'] ?? 'unknown');
        $threshold = $this->scoreThreshold();

        $isFailureStatus = in_array($status, ['auth_failed', 'dns_error', 'tls_error', 'timeout', 'rate_limited', 'unreachable'], true);
        $isLowScore = $score < $threshold;

        if (!$isFailureStatus && !$isLowScore) {
            $this->clearDedup($targetId);
            return;
        }

        $dedupKey = $this->dedupKey($targetId, $status, $isLowScore);
        if (get_transient($dedupKey)) {
            return;
        }

        $subject = sprintf('[wpOrchestrator] Alert: %s (%s)', $targetName, $status);
        $message = $this->buildMessage($target, $result, $score, $threshold);

        $emailSent = $this->sendEmail($subject, $message);
        $telegramSent = $this->sendTelegram($message);

        set_transient($dedupKey, 1, self::DEDUP_TTL);

        $this->logger->info('Alert dispatched.', [
            'target_id' => $targetId,
            'status' => $status,
            'score' => $score,
            'email_sent' => $emailSent,
            'telegram_sent' => $telegramSent,
        ]);
    }

    private function buildMessage(array $target, array $result, int $score, int $threshold): string
    {
        $lines = [
            'wpOrchestrator Alert',
            'Target: ' . (string) ($target['name'] ?? 'unknown'),
            'URL: ' . (string) ($target['base_url'] ?? 'unknown'),
            'Status: ' . (string) ($result['status'] ?? 'unknown'),
            'Message: ' . (string) ($result['message'] ?? ''),
            'Score: ' . $score . ' (threshold: ' . $threshold . ')',
            'HTTP Code: ' . (string) ($result['http_code'] ?? 0),
            'Checked at: ' . (string) ($result['checked_at'] ?? current_time('mysql')),
        ];

        return implode("\n", $lines);
    }

    private function sendEmail(string $subject, string $message): bool
    {
        $recipients = $this->emailRecipients();

        if ($recipients === []) {
            return false;
        }

        return (bool) wp_mail($recipients, $subject, $message);
    }

    private function sendTelegram(string $message): bool
    {
        $token = trim((string) get_option('wpo_telegram_bot_token', ''));
        $chatId = trim((string) get_option('wpo_telegram_chat_id', ''));
        $enabled = (string) get_option('wpo_telegram_enabled', '0') === '1';

        if (!$enabled || $token === '' || $chatId === '') {
            return false;
        }

        $endpoint = sprintf('https://api.telegram.org/bot%s/sendMessage', rawurlencode($token));
        $response = wp_remote_post($endpoint, [
            'timeout' => 12,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => wp_json_encode([
                'chat_id' => $chatId,
                'text' => $message,
                'disable_web_page_preview' => true,
            ]),
        ]);

        if (is_wp_error($response)) {
            $this->logger->error('Telegram alert failed.', ['error' => $response->get_error_message()]);
            return false;
        }

        $code = (int) wp_remote_retrieve_response_code($response);

        return $code >= 200 && $code < 300;
    }

    /**
     * @return array<int, string>
     */
    private function emailRecipients(): array
    {
        $raw = trim((string) get_option('wpo_alert_email_recipients', ''));

        if ($raw === '') {
            $admin = get_option('admin_email');
            return is_email($admin) ? [(string) $admin] : [];
        }

        $parts = array_map('trim', explode(',', $raw));
        $valid = [];

        foreach ($parts as $email) {
            if ($email !== '' && is_email($email)) {
                $valid[] = $email;
            }
        }

        return array_values(array_unique($valid));
    }

    private function scoreThreshold(): int
    {
        $value = (int) get_option('wpo_alert_score_threshold', 60);

        if ($value < 0) {
            return 0;
        }

        if ($value > 100) {
            return 100;
        }

        return $value;
    }

    private function dedupKey(int $targetId, string $status, bool $lowScore): string
    {
        $suffix = $lowScore ? 'low_score' : 'status';

        return sprintf('wpo_alert_%d_%s_%s', $targetId, sanitize_key($status), $suffix);
    }

    private function clearDedup(int $targetId): void
    {
        $statuses = ['auth_failed', 'dns_error', 'tls_error', 'timeout', 'rate_limited', 'unreachable', 'active', 'degraded'];

        foreach ($statuses as $status) {
            delete_transient(sprintf('wpo_alert_%d_%s_status', $targetId, sanitize_key($status)));
            delete_transient(sprintf('wpo_alert_%d_%s_low_score', $targetId, sanitize_key($status)));
        }
    }
}
